-- @todo a lambda to decide what's an error code.
-- @todo allow adding options to CreateProcess

module Main where

import           Prelude as P

import           System.Environment
                 ( getArgs
                 , getProgName )

import           System.Process as SP
                 ( getProcessExitCode
                 , waitForProcess )

import           Control.Concurrent
                 ( threadDelay )

import           Data.Foldable
                 ( find, traverse_ )

import           Data.Semigroup
                 ( (<>) )

import           Data.List
                 ( intercalate
                 , break
                 )

import           Control.Monad
                 ( when
                 , join
                 , void )

import           System.IO
                 ( BufferMode (NoBuffering)
                 , hSetBuffering
                 , stdout
                 , stderr
                 )

import           System.Console.Chalk
                 ( red, green )

import           Text.Speak.IO
                 ( info
                 , warn
                 , err )

import           System.Commandz.Cmd
                 ( runCmd
                 , runCmdAsync
                 , runCmdSync
                 , pipe
                 , checkCompleted
                 , checkCompletedError

                 , Exit (ExitError, ExitOk, ExitNotOk)
                 , CmdRun (Shell, Proc)
                 , Opts (Opts)
                 , Opts'
                 , ResultsSync (ResultsSyncSpawnOk, ResultsSyncSpawnFail)
                 , ResultsAsync (ResultsAsyncSpawnOk, ResultsAsyncSpawnFail)
                 , createProcessRetPHandle
                 , createProcessRetOut
                 , resultsAsyncFail
                 , resultsSyncOut
                 , resultsSyncErr
                 , resultsSyncExit
                 , optsCmdRun
                 , optsCmdRunProc
                 , optsCmdRunShell
                 , optsError
                 , optsDie
                 , optsThrow
                 , optsWarn
                 , optsIgnoreErr
                 , optsDefault
                 , optsVerbose
                 , optsVerboseNo
                 , optsVerboseYes
                 , optsInInherit
                 , optsInHandle
                 , optsInPipe
                 , optsInIgnore
                 , optsInIgnore
                 , optsOutInherit
                 , optsOutHandle
                 , optsOutPipe
                 , optsOutIgnore
                 , optsErrInherit
                 , optsErrHandle
                 , optsErrPipe
                 , optsErrIgnore
                 , optsOutPipe'
                 , optsErrPipe'
                 , optsOutCapture
                 , optsErrCapture
                 , optsOutCapture'
                 , optsErrCapture' )

import           System.Commandz.Util
                 ( (.:)
                 , (.:.)
                 , (.::)
                 , fst3
                 , snd3
                 , fth4
                 , shellQuote
                 , maybeMVoid
                 , hGetContentsStrict )

-- | die means die on either total failure or code not ok.
-- if you want to die on total failure but not on not ok, set die to false
-- and handle it with a pattern match.

unbufferOutput = traverse_ unbuffer' [stdout, stderr] where
    unbuffer' = flip hSetBuffering NoBuffering

main :: IO ()
main = do
    unbufferOutput
    args' <- getArgs
    progName' <- getProgName
    runSpec progName' args'
    pure ()

-- ignore err means don't complain in any way about a failed command;
-- err pipe / err ignore refers to stderr.

runSpec :: String -> [String] -> IO ()
runSpec progName (spec:cmd:args) = f spec specs cmd args where
    f spec specs = g (findCmd' specs)
    g Nothing _ _ = err $ usage progName
    g (Just run') cmd args = run' cmd args
    findCmd' specs = snd3 <$> find find' specs
    find' x = fst3 x == spec

runSpec progName _ = err $ usage progName

usage progName =
    "Usage: " <> progName <> " spec cmd [...args]\n" <>
    "\nwhere spec is one of:\n\n" <> specs'

specs' = intercalate "\n" . map (uncurry f) $ y specs where
    f argSpec desc = indent' $ quote' argSpec <> "\t" <> desc
    indent' = ("    " <>)
    quote' x = "'" <> x <> "'"
    y = map z
    z (a, _, b) = (a, b)

specs =
  [
    ( "s proc-die"
    , f1
    , "" )
  , ( "s proc-die"
    , f2
    , "" )
  , ( "s proc-throw"
    , f3
    , "" )
  , ( "s proc-warn"
    , f4
    , "" )
  , ("s proc-ignore-err"
    , f5
    , "" )
  , ("s proc-die-capture-err"
    , f6
    , "" )
  , ("s proc-die-capture-out"
    , f7
    , "" )
  , ("s proc-warn-capture-out"
    , f8
    , "" )

  -- we use the same setting of throw/die/warn for both the left and right
  -- commands for the pipe tests, though in principle they can use separate
  -- settings.
  , ("s connect-pipe-throw"
    , f9
    , "e.g. <main> 's connect-pipe-throw' ls -l /tmp -- wc -c")

  ] where
      f1 :: FilePath -> [String] -> IO ()
      f1 cmd args = void . runCmdSync $ cmdProcDie cmd args
      f2 = f1
      f3 cmd args = void . runCmdSync $ cmdProcThrow cmd args
      f4 cmd args = void . runCmdSync $ cmdProcWarn cmd args
      f5 cmd args = void . runCmdSync $ cmdProcIgnoreErr cmd args
      f6 cmd args = void . runCmdSync $ cmdProcDie cmd args . optsErrCapture'
      f7 cmd args = void . runCmdSync $ cmdProcDie cmd args . optsOutCapture'
      f8 cmd args = void . runCmdSync $ cmdProcWarn cmd args . optsOutCapture'
      f9 cmd args = callPipe optsThrow lCmd lArgs rCmd rArgs where
          lCmd = cmd
          -- @todo irrefutable
          (lArgs, (_:rCmd:rArgs)) = break (== "--") args

callSyncExplain opts = do
    thing' <- runCmd opts
    ok' . resultsSyncExit $ thing'
    putStrLn $ "got output: " <> show (resultsSyncOut thing')
    putStrLn $ "got error: "  <> show (resultsSyncErr thing')
    putStrLn $ "got exit: "   <> show (resultsSyncExit thing') where
        ok' ExitOk = info $ (green "✔") <> " ok"
        ok' x = putStrLn  $ (red "✘") <> " not ok: " <> show x

drainHandleIfOpen = maybeMVoid hGetContentsStrict

-- naive way to (blockingly) wait for async command to finish.
waitAsync results' = do
    done <- checkCompleted results'
    when (not done) $ do
        putStrLn "waiting"
        threadDelay 1000000
        waitAsync results'
    pure ()

callASync opts = do
    results' <- runCmd opts
    checkCompletedError results'
    -- here we block waiting for the async command, as a POC -- in real life
    -- you'll want to do something more interesting since this defeats the
    -- point of making it async in the first place.
    waitForProcess . createProcessRetPHandle $ results'
    checkCompletedError results'
    pure ()

cmdProc cmd args = optsCmdRunProc cmd args
                 . optsVerboseYes

cmdProcDie cmd args = cmdProc cmd args
                    . optsDie

cmdProcThrow cmd args = cmdProc cmd args
                      . optsThrow

cmdProcWarn cmd args = cmdProc cmd args
                     . optsWarn
                     -- @todo what was this for?
                     -- . optsCwd "/tmp"

cmdProcIgnoreErr cmd args = cmdProc cmd args
                          . optsIgnoreErr

callPipe opts lCmd lArgs rCmd rArgs = pipe left' right' where
    left' = optsCmdRunProc lCmd lArgs
          . optsVerboseYes
          . optsOutPipe'
          . opts
    right' = optsCmdRunProc rCmd rArgs
           . optsVerboseYes
           . opts

---- The rest of the file are some old functions, now superseded by the
-- external JS test suite. But they have been left here for illustrative
-- purposes.

pipeWc :: Opts' -> IO ()
pipeWc left = pipe left right' where
    right' = optsCmdRunProc "wc" ["-c"]
           . optsVerboseYes
           . optsDie

connectPipeOk = pipeWc left' where
    left' = optsCmdRunProc "echo" ["αβψ", "δεφ"]

connectPipeMissingLeftIgnore = pipeWc left' where
    left' = optsCmdRunProc "lssdfj" ["αβψ", "δεφ"]
          . optsIgnoreErr

connectPipeMissingLeftWarn = pipeWc left' where
    left' = optsCmdRunProc "lssdfj" ["αβψ", "δεφ"]
          . optsWarn

connectPipeSlowLeftOk = pipeWc left' where
    left' = optsCmdRunProc "sleep" ["2"]
          . optsThrow

connectPipeMissingLeftThrow = pipeWc left' where
    left' = optsCmdRunProc "lssdfj" ["αβψ", "δεφ"]
          . optsThrow

connectPipeMissingLeftDie = pipeWc left' where
    left' = optsCmdRunProc "lssdfj" ["αβψ", "δεφ"]
          . optsDie

connectPipeBadLeft = pipeWc left' where
    left' = optsCmdRunProc "cat" ["some-missing-file"]
          . optsWarn

cmdProcMissingThrow = optsCmdRunProc "lssdf" ["a", "b"]
                    . optsVerboseYes
                    . optsThrow

cmdProcMissingDie = optsCmdRunProc "lssdf" ["a", "b"]
                  . optsVerboseYes
                  . optsDie

cmdProcMissingIgnoreErr = optsCmdRunProc "lssdf" ["a", "b"]
                        . optsVerboseYes
                        . optsIgnoreErr

cmdProcMissingWarn  = optsCmdRunProc "lssdf" ["a", "b"]
                    . optsVerboseYes
                    . optsWarn

cmdProcBadExitThrow = optsCmdRunProc "ls" ["-l", "missing-file"]
                    . optsVerboseYes
                    . optsThrow

cmdProcBadExitDie = optsCmdRunProc "ls" ["-l", "missing-file"]
                  . optsVerboseYes
                  . optsDie

cmdProcBadExitDieErrPipe = optsCmdRunProc "ls" ["-l", "missing-file"]
                         . optsVerboseYes
                         . optsDie
                         . optsErrCapture'

cmdProcBadExitWarn = optsCmdRunProc "ls" ["-l", "missing-file"]
                   . optsVerboseYes
                   . optsWarn

cmdProcBadExitIgnoreErr = optsCmdRunProc "ls" ["-l", "missing-file"]
                        . optsVerboseYes
                        . optsIgnoreErr

cmdProcBadExitIgnoreErrErrIgnore = optsCmdRunProc "ls" ["-l", "missing-file"]
                                 . optsVerboseYes
                                 . optsIgnoreErr
                                 . optsErrIgnore

cmdProcBadExitIgnoreErrErrPipe = optsCmdRunProc "ls" ["-l", "missing-file"]
                               . optsVerboseYes
                               . optsIgnoreErr
                               . optsErrPipe'

cmdProcInPipe = optsCmdRunProc "wc" []
              . optsVerboseYes
              . optsDie
              . ( optsInPipe $ pure ["αβψ", "δεφ"] )

cmdProcOutPipe = optsCmdRunProc "find" ["/etc"]
               . optsVerboseYes
               . optsOutPipe'

cmdProcErrPipe = optsCmdRunProc "ls" ["missing-file"]
               . optsVerboseYes
               . optsDie
               . optsErrPipe'
