{-# LANGUAGE FlexibleInstances #-} -- for Dumpable String

module           System.Commandz.Cmd
                 ( runCmd
                 , runCmdAsync
                 , runCmdSync
                 , pipe
                 , checkCompleted
                 , checkCompletedCode
                 , checkCompletedError

                 -- Types
                 , SpawnRecover (spawnRecover)

                 , CmdRun (Shell, Proc)
                 , ErrorSpec (ErrorSpec, errorSpecOptsError, errorSpecCmdString)
                 , Opts (Opts, optsCmdRun, optsVerbose, optsError, optsIn, optsOut, optsErr)
                 , OptsError (OptsIgnoreErr, OptsWarn, OptsDie, OptsThrow)
                 , OptsStream (OptsInherit, OptsUse, OptsPipeWriter, OptsPipeReader, OptsIgnore)
                 , ResultsAsync (ResultsAsyncSpawnOk, createProcessRetIn,
                   createProcessRetOut, createProcessRetErr,
                   createProcessRetPHandle, createProcessRetErrorSpec,
                   ResultsAsyncSpawnFail, resultsAsyncSpawnFail)
                 , ResultsSync (ResultsSyncSpawnOk, resultsSyncExit,
                   resultsSyncOutput, resultsSyncErrorSpec,
                   ResultsSyncSpawnFail)
                 , Exit (..)
                 , CompletionException (..)

                 , Opts'
                 , CreateProcessRet'

                 , getOptsHandle
                 , isOptsHandle
                 , isOptsIgnore
                 , isOptsPipe
                 , optsCmdRunProc
                 , optsCmdRunShell
                 , optsCwd
                 , optsCwd'
                 , optsDie
                 , optsDefault
                 , optsErrCapture
                 , optsErrCapture'
                 , optsErrHandle
                 , optsErrIgnore
                 , optsErrInherit
                 , optsErrPipe
                 , optsErrPipe'
                 , optsIgnoreErr
                 , optsInHandle
                 , optsInIgnore
                 , optsInInherit
                 , optsInPipe
                 , optsOutCapture
                 , optsOutCapture'
                 , optsOutHandle
                 , optsOutIgnore
                 , optsOutInherit
                 , optsOutPipe
                 , optsOutPipe'
                 , optsShouldDie
                 , optsShouldIgnoreErr
                 , optsShouldThrow
                 , optsShouldWarn
                 , optsThrow
                 , optsVerboseNo
                 , optsVerboseYes
                 , optsWarn
                 , resultsAsyncFail
                 , resultsAsyncOk
                 , resultsSyncErr
                 , resultsSyncOut
                 , resultsSyncOutErr'

                 -- Util
                 , shellQuote ) where

import           Prelude as P

import           System.Exit as SE
                 ( ExitCode (ExitSuccess, ExitFailure) )

import           Data.Semigroup
                 ( (<>) )

import           Data.Function
                 ( (&) )

import           Data.Text as DT
                 ( pack )

import           Data.Foldable
                 ( foldl' )

import           Data.List
                 ( intercalate )

import           Data.Maybe
                 ( fromJust, isJust )

import           System.IO
                 ( Handle
                 , hPutStr
                 , hIsOpen
                 , hGetContents
                 , hClose
                 , hPutStrLn )

import           Control.Exception
                 ( Exception, IOException, SomeException, try )

import           Control.Monad
                 ( (<=<), when, void )

import           Data.String as DS
                 ( fromString )

import           Control.Monad.Trans.Except
                 ( ExceptT (ExceptT)
                 , runExceptT
                 )

import           Control.Exception
                 ( throw )

import           Control.Monad.Trans.Class
                 ( lift )

import           System.Process as SP
                 ( CreateProcess
                 , ProcessHandle
                 , StdStream (Inherit, UseHandle, NoStream, CreatePipe)
                 , getProcessExitCode
                 , cwd
                 , std_in
                 , std_out
                 , std_err
                 , waitForProcess
                 , delegate_ctlc
                 , shell
                 , createProcess_
                 , proc
                 , close_fds)

import qualified System.Commandz.Types as T
                 ( show )

import           System.Commandz.Types
                 ( Dumpable (dump)
                 , SpawnRecover (spawnRecover)

                 , CmdRun (Shell, Proc)
                 , Dump (Dump, dumpOut, dumpErr)
                 , DumpSpec (DumpYes, DumpNo)
                 , ErrorSpec (ErrorSpec, errorSpecOptsError, errorSpecCmdString)
                 , Opts (Opts, optsCmdRun, optsVerbose, optsError, optsIn, optsOut, optsErr)
                 , OptsError (OptsIgnoreErr, OptsWarn, OptsDie, OptsThrow)
                 , OptsStream (OptsInherit, OptsUse, OptsPipeWriter, OptsPipeReader, OptsIgnore)
                 , ResultsAsync (ResultsAsyncSpawnOk, createProcessRetIn,
                   createProcessRetOut, createProcessRetErr,
                   createProcessRetPHandle, createProcessRetErrorSpec,
                   ResultsAsyncSpawnFail, resultsAsyncSpawnFail)
                 , ResultsSync (ResultsSyncSpawnOk, resultsSyncExit,
                   resultsSyncOutput, resultsSyncErrorSpec,
                   ResultsSyncSpawnFail, resultsSyncExit)
                 , Exit (..)
                 , CompletionException (..)

                 , Opts'
                 , CreateProcessRet'

                 , createProcessRetOutFromJust
                 , errorSpecMapString
                 , getOptsHandle
                 , isOptsHandle
                 , isOptsIgnore
                 , isOptsPipe
                 , makeExit
                 , mkDumper
                 , optsCmdRunProc
                 , optsCmdRunShell
                 , optsCwd
                 , optsCwd'
                 , optsDie
                 , optsDefault
                 , optsErrCapture
                 , optsErrCapture'
                 , optsErrHandle
                 , optsErrIgnore
                 , optsErrInherit
                 , optsErrPipe
                 , optsErrPipe'
                 , optsIgnoreErr
                 , optsInHandle
                 , optsInIgnore
                 , optsInInherit
                 , optsInPipe
                 , optsOutCapture
                 , optsOutCapture'
                 , optsOutHandle
                 , optsOutIgnore
                 , optsOutInherit
                 , optsOutPipe
                 , optsOutPipe'
                 , optsShouldDie
                 , optsShouldIgnoreErr
                 , optsShouldThrow
                 , optsShouldWarn
                 , optsThrow
                 , optsVerboseNo
                 , optsVerboseYes
                 , optsWarn
                 , resultsAsyncFail
                 , resultsAsyncOk
                 , resultsSyncErr
                 , resultsSyncOut
                 , resultsSyncOutErr'
                 , runDumper
                 , upgradeCreateProcessRet )

import           System.Commandz.Util
                 ( (.:), (.:.)
                 , (<*.)
                 , (>>.)
                 , fth4
                 , makeExceptT
                 , getOptsCmd
                 , shellQuote
                 , rejectEmpty
                 , maybeMVoid
                 , hGetContentsStrict )

import           System.Commandz.Die
                 ( dieWithCode )

import           Text.Speak
                 ( bullet
                 , infoStr
                 , errStr )

import           Text.Speak.IO
                 ( err
                 , warn
                 )

import           System.Console.Chalk
                 ( cyan
                 , green
                 , yellow )

import           Text.Strung as S
                 ( IsStrung
                 , fromString
                 , intercalate )

spawnErrStr    = "Couldn't spawn command"

-- The polymorphism provided by this class is how you run the command after
-- assembling it. To run it asynchronously, use `runCmdAsync`, or use
-- `runCmd` and force the result to have type `IO ResultsAsync`. For
-- synchronous, use `runCmdSync` or use `runCmd` and force the type `IO
-- ResultsSync`.

class RunCmd a where
    runCmd :: Opts' -> IO a

instance RunCmd ResultsAsync where
    runCmd = runCmdAsync

instance RunCmd ResultsSync where
    runCmd = runCmdSync

runCmdAsync :: Opts' -> IO ResultsAsync
runCmdAsync opts = do
    let opts'       = opts optsDefault
        throw'      = optsShouldThrow opts'
        die'        = optsShouldDie opts'
        warn'       = optsShouldWarn opts'
        spawnError' = spawnError throw' die' warn'
        run'        = runExceptT . run $ opts
        run         = runCmd'

    eitExit'   <- run' :: IO (Either SomeException ResultsAsync)
    either spawnError' pure eitExit'

runCmdSync :: Opts' -> IO ResultsSync
runCmdSync opts = do
    let opts'       = opts optsDefault
        throw'      = optsShouldThrow opts'
        die'        = optsShouldDie opts'
        warn'       = optsShouldWarn opts'
        spawnError' = spawnError throw' die' warn'
        run'        = runExceptT . runCmdSync' $ opts

    eitExit'   <- run' :: IO (Either SomeException ResultsSync)
    results' <- either spawnError' pure eitExit'

    let exit'  = resultsSyncExit results'
        errSpec = resultsSyncErrorSpec results'
        (out', err') = resultsSyncOutput results'

        errSpec' = errorSpecMapString f errSpec
        mbcode = checkCompletedWithError exit'
        f s = "᚜" <> s <> "᚛ " <> show exit'
        dump = mkDumper dumpOut dumpErr out' err'

        -- @todo streams are currently always dumped if the command is not
        -- successful.
        dumpOut = True
        dumpErr = True

    checkCompletionError' dump errSpec' mbcode
    pure results'

checkCompletedCode :: ResultsAsync -> IO (Maybe ExitCode)
checkCompletedCode (ResultsAsyncSpawnFail _) = error e' where
    e' = "checkCompletedCode called on process which never spawned"
checkCompletedCode ret = getProcessExitCode . createProcessRetPHandle $ ret

checkCompleted :: ResultsAsync -> IO Bool
checkCompleted ret = isJust <$> checkCompletedCode ret

checkCompletedError :: ResultsAsync -> IO ()
checkCompletedError (ResultsAsyncSpawnFail _) = error e' where
    e' = "checkCompletedError called on process which never spawned"
checkCompletedError ret@(ResultsAsyncSpawnOk _ mbOut' mbErr' pHandle' errStr') = do
    checkCompletionError' dumper' errStr' =<< checkCompletedCode ret where
        dumper' = mkDumper dumpOut dumpErr mbOut' mbErr'
        -- @todo streams are currently always dumped if the command is not
        -- successful.
        dumpOut = True
        dumpErr = True

instance Dumpable String where
    dump = putStrLn

instance Dumpable Handle where
    dump = f <=< hGetContentsStrictIfOpen where
        f Nothing = pure ()
        f (Just s) = putStrLn s

hGetContentsStrictIfOpen :: Handle -> IO (Maybe String)
hGetContentsStrictIfOpen h = f =<< hIsOpen h where
    f True = Just <$> hGetContentsStrict h
    f False = pure Nothing

-- Results in pure () if the code is Nothing (command not completed). If the
-- code is Just (command has completed) then results in pure () (ignore),
-- pure () and a warning (warn), an exception (throw) or die (die).

checkCompletionError' :: Dumpable a => Dump a -> ErrorSpec -> Maybe ExitCode -> IO ()
checkCompletionError' dump' errorSpec' = maybeMVoid check' where
    check' code' = when (isError' code') $ cleanup' code'
    cleanup' code' = runDumper dump' >> completionError error' str' code'
    ErrorSpec error' str' = errorSpec'
    isError' (ExitFailure _) = True
    isError' _ = False

completionError :: IsStrung a => OptsError -> a -> ExitCode -> IO ()
completionError = c' where
    c' OptsIgnoreErr _ _  = pure ()
    c' OptsWarn str code  = warn str
    c' OptsThrow str code = throw $ CompletionException code str
    c' OptsDie str code   = dieWithCode code str

spawnError :: (SpawnRecover a, Exception e) => Bool -> Bool -> Bool -> e -> IO a
spawnError throw' die' warn' = f where
    f   | throw'    = throw
        | die'      = die''
        | otherwise = g
    w e | warn' = warn $ show e
        | otherwise = pure ()
    g = w >>. recover'
    recover' = pure . spawnRecover
    -- the 'recover' is never reached and is just there to satisfy the 'IO a'
    -- requirement.
    die'' e = (dieWithCode (ExitFailure 1) . show $ e) *> recover' e

checkExitNotOk, checkExitOk :: Exit -> Bool
checkExitNotOk     = not . checkExitOk
checkExitOk ExitOk = True
checkExitOk _      = False

checkCompletedWithError :: Exit -> Maybe ExitCode
checkCompletedWithError (ExitNotOk c) = Just c
checkCompletedWithError _             = Nothing

runCmd' :: Opts' -> ExceptT SomeException IO ResultsAsync
runCmd' opts = do
    let opts'      = opts optsDefault
        cmd'       = optsCmdRun opts'
        verbose'   = optsVerbose opts'
        throw'     = optsShouldThrow opts'
        die'       = optsShouldDie opts'
        warn'      = optsShouldWarn opts'
        transform' = getStreamTransform opts'
        cwd'       = optsCwd' opts'
        create'    = transform' $ create cmd' cwd'
        run        = tryCreateProcess spawnErrStr create'
        cmdStr'    = getOptsCmd opts'
        error'     = optsError opts'
        errorSpec' = ErrorSpec error' cmdStr'
    when verbose' . lift $ printCmd opts'
    upgradeCreateProcessRet errorSpec' <$> makeExceptT run

getStreamTransform :: Opts -> CreateProcess -> CreateProcess
getStreamTransform opts' = transform transform' where

    inPipe'          = isOptsPipe               $ optsIn opts'
    outPipe'         = isOptsPipe               $ optsOut opts'
    errPipe'         = isOptsPipe               $ optsErr opts'
    inIgnore'        = isOptsIgnore             $ optsIn opts'
    outIgnore'       = isOptsIgnore             $ optsOut opts'
    errIgnore'       = isOptsIgnore             $ optsErr opts'
    inHandle'        = isOptsHandle             $ optsIn opts'
    outHandle'       = isOptsHandle             $ optsOut opts'
    errHandle'       = isOptsHandle             $ optsErr opts'
    inHandleHandle'  = fromJust . getOptsHandle $ optsIn opts'
    outHandleHandle' = fromJust . getOptsHandle $ optsOut opts'
    errHandleHandle' = fromJust . getOptsHandle $ optsErr opts'

    transform' = [ (inPipe', std_in_createPipe)
                 , (outPipe', std_out_createPipe)
                 , (errPipe', std_err_createPipe)
                 , (inIgnore', std_in_ignore)
                 , (outIgnore', std_out_ignore)
                 , (errIgnore', std_err_ignore)
                 , (inHandle', std_in_handle inHandleHandle')
                 , (outHandle', std_out_handle outHandleHandle')
                 , (errHandle', std_err_handle errHandleHandle') ]

    transform :: [(Bool, a -> a)] -> a -> a
    transform ts x = foldl' f x ts where
        f x (True, g) = g x
        f x (False, _) = x

runCmdSync' :: Opts' -> ExceptT SomeException IO ResultsSync
runCmdSync' opts = do
    let opts'      = opts optsDefault
        wantsInput'    = isOptsPipe $ optsIn opts'
        hasCapturedOut'   = isOptsPipe $ optsOut opts'
        hasCapturedErr'   = isOptsPipe $ optsErr opts'
        stream' True mb' = Just mb'
        stream' False _ = Nothing
        out' = stream' hasCapturedOut'
        err' = stream' hasCapturedErr'
    ResultsAsyncSpawnOk mbIn' mbOut' mbErr' pHandle' errStr' <- runCmd' opts
    when wantsInput' . lift  $ writeInput opts' mbIn'
    lift                 $ completeCmd mbOut' mbErr' pHandle' errStr'

-- | First drain the pipes, then wait for the process to finish.
--   GHC Note: in order to call waitForProcess without blocking all the
--   other threads, you must compile the program with -threaded.

completeCmd :: Maybe Handle -> Maybe Handle -> ProcessHandle -> ErrorSpec -> IO ResultsSync
completeCmd mbOut mbErr pHandle errorSpec = do
    let slurp' :: Maybe Handle -> Maybe (IO String)
        slurp  :: Handle -> IO String
        slurp' = fmap slurp
        slurp  = hGetContentsStrict <*. hClose

        mbOut' = slurp' mbOut :: Maybe (IO String)
        mbErr' = slurp' mbErr :: Maybe (IO String)

    io <- seqMaybeIO mbOut' mbErr' :: IO (Maybe String, Maybe String)
    exit <- makeExit <$> waitForProcess pHandle
    pure $ ResultsSyncSpawnOk exit io errorSpec

seqMaybeIO :: Maybe (IO String) -> Maybe (IO String) -> IO (Maybe String, Maybe String)
seqMaybeIO a b = (,) <$> f a <*> f b where
    f (Just y) = pure . Just =<< y
    f Nothing = pure Nothing

std_in_createPipe  o = o { std_in  = CreatePipe }
std_out_createPipe o = o { std_out = CreatePipe }
std_err_createPipe o = o { std_err = CreatePipe }
std_in_ignore      o = o { std_in  = NoStream }
std_out_ignore     o = o { std_out = NoStream }
std_err_ignore     o = o { std_err = NoStream }
std_in_handle    h o = o { std_in  = UseHandle h }
std_out_handle   h o = o { std_out = UseHandle h }
std_err_handle   h o = o { std_err = UseHandle h }

writeInput :: Opts -> Maybe Handle -> IO ()
writeInput opts' mbIn' = do
    let fp = getInPipe mbIn'
    mapM_ (hPutStr fp) =<< getInStream opts'
    hClose fp

getInStream :: Opts -> IO [String]
getInStream = get' . optsIn where
    get' (OptsPipeWriter w) = w
    get' _ = error "getInStream: bad call"

-- Defend against broken external interface (streams should not be Nothing
-- if `CreatePipe` was used)

getInPipe :: Maybe Handle -> Handle
getInPipe = getPipeStream' "input" where
    getPipeStream' _ (Just x) = x
    getPipeStream' which Nothing = error $ "getPipeStream': expected " <> which <> " pipe"

-- `delegate_ctlc` seems to be counter-intuitive:
-- `False` (default) means the subprocess gets the signal.

create :: CmdRun -> Maybe FilePath -> CreateProcess
create (Shell s) cwdMb = (shell s) { delegate_ctlc = False
                                   , cwd = cwdMb }
create (Proc f a) cwdMb = (proc f a) { delegate_ctlc = False
                                     , cwd = cwdMb }

-- We use `createProcess_` so that the handles stay open, and so that we can
-- provide a custom error string.
tryCreateProcess :: String -> CreateProcess -> IO (Either SomeException CreateProcessRet')
tryCreateProcess = try .: createProcess_

printCmd' :: IsStrung a => Maybe FilePath -> CmdRun -> (Bool, Bool, Bool) -> a
printCmd' cwdMb cmdRun handles = joinNL . concat $ print' where
    print' = [ printCwd' cwdMb, [ printCmd'' cmdRun handles ] ]
    printCwd' Nothing = []
    printCwd' (Just fp) = [infoStr . S.fromString $ "[ " <> yellow "chdir" <> " ] " <> fp]
    printCmd'' (Shell s) = fmt id s []
    printCmd'' (Proc f a) = fmt cyan f a
    fmt color' h t = fmt' (head' color' h) (tail' t)
    fmt' = DS.fromString .:. str'
    head' color' = color' . shellQuote
    tail' = map shellQuote
    arg2' x y = S.intercalate " " . rejectEmpty $ [x, y]
    arg3' x y z = arg2' x $ arg2' y z
    outPipe' True = "|"
    outPipe' False = ""
    errPipe' True = "2|"
    errPipe' False = ""
    joinNL = S.intercalate . S.fromString $ "\n"
    -- @todo possibly too rigid in the type.
    join = S.intercalate . S.fromString $ " "
    bullet' = green bullet <> " "
    str' h t (pipeIn', pipeOut', pipeErr') =
        bullet' <> arg2' (outPipe' pipeIn') h <>
        " "     <> arg3' (join t) (outPipe' pipeOut') (errPipe' pipeErr')

printCmd :: Opts -> IO ()
printCmd opts' = putStrLn print' where
    cwd'       = optsCwd' opts'
    print'     = printCmd' cwd' cmdRun' spec'
    isPipeIn'  = opts' & isOptsPipe   . optsIn
    isPipeOut' = opts' & isOptsPipe   . optsOut
    isPipeErr' = opts' & isOptsPipe   . optsErr
    isUseIn'   = opts' & isOptsHandle . optsIn
    isUseOut'  = opts' & isOptsHandle . optsOut
    isUseErr'  = opts' & isOptsHandle . optsErr
    cmdRun'    = opts' & optsCmdRun
    spec'      = ( isPipeIn'  || isUseIn'
                 , isPipeOut' || isUseOut'
                 , isPipeErr' || isUseErr' )

pipe :: Opts' -> Opts' -> IO ()
pipe left' right' = connectPipe right' =<< runCmdAsync left'

connectPipe :: Opts' -> ResultsAsync -> IO ()
connectPipe cmdRight retLeft = pipe' leftFailed' where
    pipe' True = pure ()
    pipe' False = runRight' outLeft' >> checkLeft'

    leftFailed' = resultsAsyncFail retLeft
    outLeft' = createProcessRetOutFromJust retLeft
    runRight' = runCmdSync . cmdRight'
    checkLeft' = checkCompletedError retLeft
    cmdRight' out' = cmdRight . optsInHandle out'
