{-# LANGUAGE GADTs #-}
{-# LANGUAGE PackageImports #-}

module           System.Commandz.Types
                 ( Dumpable (..)
                 , SpawnRecover (..)
                 , show

                 , CmdRun (..)
                 , Dump (..)
                 , DumpSpec (..)
                 , ErrorSpec (..)
                 , Opts (..)
                 , OptsError (..)
                 , OptsStream (..)
                 , ResultsAsync (..)
                 , ResultsSync (..)

                 , Exit (..)
                 , CompletionException (..)

                 , Opts'
                 , CreateProcessRet'

                 , createProcessRetOutFromJust
                 , errorSpecMapString
                 , getOptsHandle
                 , isOptsHandle
                 , isOptsIgnore
                 , isOptsPipe
                 , makeExit
                 , mkDumper
                 , optsCmdRunProc
                 , optsCmdRunShell
                 , optsCwd
                 , optsDie
                 , optsDefault
                 , optsErrCapture
                 , optsErrCapture'
                 , optsErrHandle
                 , optsErrIgnore
                 , optsErrInherit
                 , optsErrPipe
                 , optsErrPipe'
                 , optsIgnoreErr
                 , optsInHandle
                 , optsInIgnore
                 , optsInInherit
                 , optsInPipe
                 , optsOutCapture
                 , optsOutCapture'
                 , optsOutHandle
                 , optsOutIgnore
                 , optsOutInherit
                 , optsOutPipe
                 , optsOutPipe'
                 , optsShouldDie
                 , optsShouldIgnoreErr
                 , optsShouldThrow
                 , optsShouldWarn
                 , optsThrow
                 , optsVerboseNo
                 , optsVerboseYes
                 , optsWarn
                 , resultsAsyncFail
                 , resultsAsyncOk
                 , resultsSyncErr
                 , resultsSyncOut
                 , resultsSyncOutErr'
                 , runDumper
                 , upgradeCreateProcessRet )
                 where

import           System.Exit as SE
                 ( ExitCode (ExitSuccess, ExitFailure) )

import           System.Process as SP
                 ( ProcessHandle )

import           System.IO
                 ( Handle )

import           Data.List as DL
                 ( intercalate )

import           Data.Semigroup
                 ( (<>) )

import           Data.Maybe
                 ( fromJust )

import           Control.Exception
                 ( Exception )

import qualified Data.List as DL
                 ( find, intercalate )

import           Data.Semigroup
                 ( Semigroup )

import           Text.Strung
                 ( IsStrung
                 , toString )

data CmdRun = Shell String
            | Proc FilePath [String]
            deriving Show

data OptsError = OptsIgnoreErr | OptsWarn | OptsDie | OptsThrow

data Opts = Opts { optsCmdRun :: CmdRun
                 , optsCwd' :: Maybe FilePath
                 , optsVerbose :: Bool
                 , optsError :: OptsError
                 , optsIn :: OptsStream
                 , optsOut :: OptsStream
                 , optsErr :: OptsStream }

data OptsStream = OptsInherit
                | OptsUse Handle
                -- String because Handle takes String
                | OptsPipeWriter (IO [String])
                | OptsPipeReader ([String] -> IO String)
                | OptsIgnore

optsDefault = Opts { optsCmdRun = Shell ""
                   , optsCwd' = Nothing
                   , optsVerbose = True
                   , optsError = OptsWarn
                   , optsIn = OptsInherit
                   , optsOut = OptsInherit
                   , optsErr = OptsInherit }

isOptsPipe (OptsPipeWriter _) = True
isOptsPipe (OptsPipeReader _) = True
isOptsPipe _ = False

isOptsIgnore OptsIgnore = True
isOptsIgnore _ = False

isOptsHandle (OptsUse _) = True
isOptsHandle _ = False

getOptsHandle (OptsUse h) = Just h
getOptsHandle _ = Nothing

-- @todo not great: ignoreErr means keep going after an error and don't
-- warn; errIgnore means kill the stream.

optsVerboseYes     o = o { optsVerbose = True }
optsVerboseNo      o = o { optsVerbose = False }
optsCmdRunShell  s o = o { optsCmdRun  = Shell s }
optsCmdRunProc f a o = o { optsCmdRun  = Proc f a }
optsCwd          c o = o { optsCwd'    = Just c }
optsDie            o = o { optsError   = OptsDie }
optsThrow          o = o { optsError   = OptsThrow }
optsWarn           o = o { optsError   = OptsWarn }
optsIgnoreErr      o = o { optsError   = OptsIgnoreErr }
optsInInherit      o = o { optsIn      = OptsInherit }
optsInHandle     h o = o { optsIn      = OptsUse h }
optsInPipe       w o = o { optsIn      = OptsPipeWriter w }
optsInIgnore       o = o { optsIn      = OptsIgnore }
optsOutInherit     o = o { optsOut     = OptsInherit }
optsOutHandle    h o = o { optsOut     = OptsUse h }
optsOutPipe      r o = o { optsOut     = OptsPipeReader r }
optsOutIgnore      o = o { optsOut     = OptsIgnore }
optsErrInherit     o = o { optsErr     = OptsInherit }
optsErrHandle    h o = o { optsErr     = OptsUse h }
optsErrPipe      r o = o { optsErr     = OptsPipeReader r }
optsErrIgnore      o = o { optsErr     = OptsIgnore }

-- out pipe + sync means read the whole thing
-- out pipe + no sync means give the handle back. they can then read it or
-- provide it via usehandle to a new (piped input) process.

optsShouldDie = o' . optsError where
    o' OptsDie = True
    o' _ = False

optsShouldWarn      = o' . optsError where
    o' OptsWarn = True
    o' _ = False

optsShouldThrow = o' . optsError where
    o' OptsThrow = True
    o' _ = False

optsShouldIgnoreErr = o' . optsError where
    o' OptsIgnoreErr = True
    o' _ = False

data CompletionException where
    CompletionException :: IsStrung a => ExitCode -> a -> CompletionException

instance Exception CompletionException

instance Show CompletionException where
    show (CompletionException code str) = toString str

data SpawnException where
    SpawnException :: Exception e => e -> SpawnException

instance Exception SpawnException

instance Show SpawnException where
    show (SpawnException e) = "SpawnException: " <> show e

class SpawnRecover a where
    spawnRecover :: Exception e => e -> a

instance SpawnRecover ResultsSync where
    spawnRecover = ResultsSyncSpawnFail . ExitError

instance SpawnRecover ResultsAsync where
    spawnRecover = ResultsAsyncSpawnFail . SpawnException

data Exit where
    ExitOk    :: Exit
    ExitNotOk :: ExitCode -> Exit
    ExitError :: Exception e => e -> Exit

instance Show Exit where
    show ExitOk        = "Command completed successfully"
    show (ExitError e) = "Unable to spawn command: " <> show e
    show (ExitNotOk c) = "Command completed with non-zero exit code: " <> show c

data ResultsAsync = ResultsAsyncSpawnOk { createProcessRetIn :: Maybe Handle
                                        , createProcessRetOut :: Maybe Handle
                                        , createProcessRetErr :: Maybe Handle
                                        , createProcessRetPHandle :: ProcessHandle
                                        , createProcessRetErrorSpec :: ErrorSpec }
                  | ResultsAsyncSpawnFail { resultsAsyncSpawnFail :: SpawnException }

createProcessRetOutFromJust = fromJust . createProcessRetOut

resultsAsyncOk :: ResultsAsync -> Bool
resultsAsyncOk (ResultsAsyncSpawnFail _) = False
resultsAsyncOk (ResultsAsyncSpawnOk _ _ _ _ _) = True

resultsAsyncFail :: ResultsAsync -> Bool
resultsAsyncFail = not . resultsAsyncOk

type CreateProcessRet' = (Maybe Handle, Maybe Handle, Maybe Handle, ProcessHandle)
type Opts' = Opts -> Opts

data ErrorSpec = ErrorSpec { errorSpecOptsError :: OptsError
                           , errorSpecCmdString :: String }

errorSpecMapString f spec = spec { errorSpecCmdString = g spec } where
    g = f . errorSpecCmdString

-- | Represents a completed command, so there is no use for a stdin handle.
--   `resultsSyncErrorSpec` contains the information about whether it
--   completed successfully.

data ResultsSync = ResultsSyncSpawnOk { resultsSyncExit :: Exit
                                      , resultsSyncOutput :: (Maybe String, Maybe String)
                                      , resultsSyncErrorSpec :: ErrorSpec}
                 | ResultsSyncSpawnFail { resultsSyncExit :: Exit }

-- | A safe version of `resultsSyncOutput`, which fills in the hole in the
--   sum type variant.

resultsSyncOutErr' :: ResultsSync -> (Maybe String, Maybe String)
resultsSyncOutErr' (ResultsSyncSpawnOk _ oe _) = oe
resultsSyncOutErr' _ = (Nothing, Nothing)

-- | These get the output and error streams of a completed sync command.
--   Caller must be sure that the command spawned properly and that the
--   output/error was captured, or else it will result in a runtime error.

resultsSyncOut, resultsSyncErr :: ResultsSync -> String
resultsSyncOut = fromJust . fst . resultsSyncOutErr'
resultsSyncErr = fromJust . snd . resultsSyncOutErr'

-- | More convenient than the generic form: return a string with all the
-- output / error.

optsOutPipe', optsErrPipe' :: Opts -> Opts
optsOutPipe' = optsOutPipe $ pure . intercalate ""
optsErrPipe' = optsErrPipe $ pure . intercalate ""

-- synonyms for out/err pipes
optsOutCapture       = optsOutPipe
optsErrCapture       = optsErrPipe
optsOutCapture'      = optsOutPipe'
optsErrCapture'      = optsErrPipe'

makeExit SE.ExitSuccess = ExitOk
makeExit (SE.ExitFailure c) = ExitNotOk $ ExitFailure c

upgradeCreateProcessRet err' (a, b, c, d) = ResultsAsyncSpawnOk a b c d err'

data Dump a = Dump { dumpOut :: DumpSpec a
                   , dumpErr :: DumpSpec a }

data DumpSpec a = DumpYes a
                | DumpNo

mkDumper :: Bool -> Bool -> Maybe a -> Maybe a -> Dump a
mkDumper doO doE o e = Dump (f doO o) (f doE e) where
    f True (Just x) = DumpYes x
    f _ _           = DumpNo

runDumper :: Dumpable a => Dump a -> IO ()
runDumper (Dump o e) = f o >> f e where
    f (DumpYes x) = dump x
    f DumpNo = pure ()

class Dumpable a where
    dump :: a -> IO ()
