module           Text.Speak.IO
                 ( blue'
                 , brightBlue'
                 , brightCyan'
                 , brightGreen'
                 , brightMagenta'
                 , brightRed'
                 , brightYellow'
                 , cmdPref'
                 , cyan'
                 , err
                 , green'
                 , info
                 , magenta'
                 , red'
                 , underline'
                 , warn
                 , wrap
                 , yellow' ) where

import           Prelude hiding
                 ( putStr )

import           Control.Conditional
                 ( ifM )

import           Control.Monad
                 ( (<=<) )

import           System.IO
                 ( hIsTerminalDevice
                 , stderr
                 , stdout )

import           System.Console.Chalk
                 ( underline
                 , bold
                 , red
                 , green
                 , yellow
                 , blue
                 , magenta
                 , cyan )

import           Text.Strung
                 ( IsStrung
                 , fromString
                 , hPutStr
                 , putStr )

import           Text.Speak.Internal
                 ( appendNL
                 , join3
                 , pref )

-- | Wrap single-arity formatting functions (e.g. colors, underline) with a
-- check to see if stdout is connected to a TTY; if not, the strings will be
-- left unchanged by the resulting function.

wrap :: (a -> a) -> IO (a -> a)
wrap f = ifTTYM (pure f) (pure id)

ifTTYM :: IO a -> IO a -> IO a
ifTTYM = ifM isStdoutTTY

isStdoutTTY :: IO Bool
isStdoutTTY = hIsTerminalDevice stdout

-- | versions which check if stdout is a TTY.

underline' :: IsStrung a => IO (a -> a)
underline' = wrap underline

red' :: IsStrung a => IO (a -> a)
red' = wrap red

green' :: IsStrung a => IO (a -> a)
green' = wrap green

yellow' :: IsStrung a => IO (a -> a)
yellow' = wrap yellow

blue' :: IsStrung a => IO (a -> a)
blue' = wrap blue

magenta' :: IsStrung a => IO (a -> a)
magenta' = wrap magenta

cyan' :: IsStrung a => IO (a -> a)
cyan' = wrap cyan

brightRed' :: IsStrung a => IO (a -> a)
brightRed' = wrap (bold . red)

brightGreen' :: IsStrung a => IO (a -> a)
brightGreen' = wrap (bold . green)

brightYellow' :: IsStrung a => IO (a -> a)
brightYellow' = wrap (bold . yellow)

brightBlue' :: IsStrung a => IO (a -> a)
brightBlue' = wrap (bold . blue)

brightMagenta' :: IsStrung a => IO (a -> a)
brightMagenta' = wrap (bold . magenta)

brightCyan' :: IsStrung a => IO (a -> a)
brightCyan' = wrap (bold . cyan)

cmdPref' :: IsStrung a => IO a
cmdPref' = pref <$> wrap green

warnPref' :: IsStrung a => IO a
warnPref' = pref <$> wrap (bold . red)

errPref' :: IsStrung a => IO a
errPref'  = pref <$> wrap red

infoPref' :: IsStrung a => IO a
infoPref' = pref <$> wrap blue

fromString' :: IsStrung a => String -> IO a
fromString' = pure . fromString

errStr' :: IsStrung a => a -> IO a
errStr' s = join3 <$> errPref' <*> fromString' "Error: " <*> pure s

warnStr' :: IsStrung a => a -> IO a
warnStr' s = join3 <$> warnPref' <*> fromString' "Warning: " <*> pure s

infoStr' :: IsStrung a => a -> IO a
infoStr' s = (<>) <$> infoPref' <*> pure s

err' :: IsStrung a => a -> IO ()
err' = hPutStr stderr <=< errStr'

warn' :: IsStrung a => a -> IO ()
warn' = hPutStr stderr <=< warnStr'

info' :: IsStrung a => a -> IO ()
info' = putStr <=< infoStr'

warn :: IsStrung a => a -> IO ()
warn = warn' . appendNL

err :: IsStrung a => a -> IO ()
err  = err' . appendNL

info :: IsStrung a => a -> IO ()
info = info' . appendNL
