-- | IsStrung is based on IsString, extended with more methods and
--   'superclasses'. The main use is to write library functions which take
--   both String and Text. We recommend *not* using OverloadedStrings when
--   working with those kinds of functions.

{-# LANGUAGE FlexibleInstances #-} -- for [Char] as instance

module           Text.Strung
                 ( IsStrung
                 , filter
                 , find
                 , foldr
                 , fromString
                 , hPutStr
                 , intercalate
                 , putStr
                 , show
                 , toString ) where

import           Prelude
                 hiding ( foldr, show, putStr )

import qualified Prelude as P
                 ( break
                 , concat
                 , concatMap
                 , drop
                 , filter
                 , foldr
                 , map
                 , putStr
                 , show
                 , tail
                 , take
                 , unlines
                 , unwords
                 , words
                 , zip
                 , zipWith
                 )

import qualified Data.List as DL
                 ( find
                 , findIndex
                 , intercalate
                 , intersperse
                 , reverse
                 , tails
                 , uncons
                 , unfoldr
                 )

import qualified Data.Foldable as DF
                 ( foldl' )

import           Text.Printf
                 ( PrintfArg )

import           Data.Semigroup
                 ( (<>) )

import qualified Data.String as DS
                 ( fromString
                 , lines )

import           Data.String as DS
                 ( IsString )

import qualified Data.Text as DT
                 ( break
                 , concat
                 , concatMap
                 , drop
                 , filter
                 , find
                 , findIndex
                 , foldl'
                 , foldr
                 , intercalate
                 , intersperse
                 , lines
                 , map
                 , reverse
                 , snoc
                 , tail
                 , tails
                 , take
                 , uncons
                 , unfoldr
                 , unlines
                 , unwords
                 , words
                 , zip
                 , zipWith
                 )

import           Data.Text as DT
                 ( Text
                 , cons
                 , unpack
                 , pack )

import qualified Data.Text.IO as DTIO
                 ( putStr
                 , hPutStr )

import           System.IO as SIO
                 ( Handle )

import qualified System.IO as SIO
                 ( hPutStr )

-- | `Monoid` is here to make System.Console.Chalk (chalk) happy, though
-- Semigroup would probably have been good enough.

class (Show a, Monoid a, PrintfArg a, IsString a) => IsStrung a where
    break       :: (Char -> Bool) -> a -> (a, a)
    concat      :: [a] -> a
    concatMap   :: (Char -> a) -> a -> a
    cons        :: Char -> a -> a
    drop        :: Int -> a -> a
    find        :: (Char -> Bool) -> a -> Maybe Char
    findIndex   :: (Char -> Bool) -> a -> Maybe Int
    -- the fold definitions are slightly lax in the 3rd argument.
    foldl'      :: (b -> Char -> b) -> b -> a -> b
    foldr       :: (Char -> b -> b) -> b -> a -> b
    fromString  :: String -> a
    hPutStr     :: Handle -> a -> IO ()
    intercalate :: a -> [a] -> a
    intersperse :: Char -> a -> a
    lines       :: a -> [a]
    map         :: (Char -> Char) -> a -> a
    putStr      :: a -> IO ()
    reverse     :: a -> a
    show        :: Show b => b -> a
    snoc        :: a -> Char -> a
    tail        :: a -> a
    tails       :: a -> [a]
    take        :: Int -> a -> a
    toString    :: a -> String
    uncons      :: a -> Maybe (Char, a)
    unfoldr     :: (b -> Maybe (Char, b)) -> b -> a
    unlines     :: [a] -> a
    unwords     :: [a] -> a
    words       :: a -> [a]
    zip         :: a -> a -> [(Char, Char)]
    zipWith     :: (Char -> Char -> Char) -> a -> a -> a

instance IsStrung Text where
    break       = DT.break
    concat      = DT.concat
    concatMap   = DT.concatMap
    cons        = DT.cons
    drop        = DT.drop
    find        = DT.find
    findIndex   = DT.findIndex
    foldl'      = DT.foldl'
    foldr       = DT.foldr
    fromString  = DS.fromString
    hPutStr     = DTIO.hPutStr
    intercalate = DT.intercalate
    intersperse = DT.intersperse
    lines       = DT.lines
    map         = DT.map
    putStr      = DTIO.putStr
    reverse     = DT.reverse
    show        = DS.fromString . P.show
    snoc        = DT.snoc
    tail        = DT.tail
    tails       = DT.tails
    take        = DT.take
    toString    = unpack
    uncons      = DT.uncons
    unfoldr     = DT.unfoldr
    unlines     = DT.unlines
    unwords     = DT.unwords
    words       = DT.words
    zip         = DT.zip
    zipWith     = DT.zipWith

-- Also FilePath, which is a synonym for String.

instance IsStrung String where
    break       = P.break
    concat      = P.concat
    concatMap   = P.concatMap
    cons        = (:)
    drop        = P.drop
    find        = DL.find
    findIndex   = DL.findIndex
    foldl'      = DF.foldl'
    foldr       = P.foldr
    fromString  = DS.fromString
    hPutStr     = SIO.hPutStr
    intercalate = DL.intercalate
    intersperse = DL.intersperse
    lines       = DS.lines
    map         = P.map
    putStr      = P.putStr
    reverse     = DL.reverse
    show        = P.show
    snoc xs x   = xs ++ [x]
    tail "" = error "tail: empty"
    tail (_ : xs) = xs
    tails       = DL.tails
    take        = P.take
    toString    = id
    uncons      = DL.uncons
    unfoldr     = DL.unfoldr
    unlines     = P.unlines
    unwords     = P.unwords
    words       = P.words
    zip         = P.zip
    zipWith     = P.zipWith
